<?php
//__NM____NM__FUNCTION__NM__//

class JsonCommunicator
{
	protected $validRequest;
	protected $responseStatus = 200;
	protected $headersSet = false;
	protected $request = [];
	protected $userData = false;
	
	public function __construct($acc_request) {
		$this->validRequest = (in_array($_SERVER['REQUEST_METHOD'], $acc_request));
		$this->setRequest();
		$this->gatherUserData();
	}
	
	public function apiWrapUp($require_auth = true) {
		if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
			$this->sendOK();
		} else {
			if (!$this->validRequest){
				$this->sendBadRequest();
			} else if ($require_auth) {
				if (!$this->userId()) {
					$this->sendForbidden();
				}
			}
		}
		$this->setHeadersOnce();
	}

	public function sendOK() {
		$this->setStatus(200);
		$this->sendResponse([
			'status' => 'OK'
		]);
	}

	public function sendBadRequest($msg = false) {
		$this->setStatus(400);
		$this->sendResponse([
			'status' => 'BAD_REQUEST',
			'msg' => $msg ?: 'Tipo de requisição ('.$_SERVER['REQUEST_METHOD'].') inválido!'
		]);
	}

	public function sendForbidden($msg = false) {
		$this->setStatus(403);
		$this->sendResponse([
			'status' => 'FORBIDDEN',
			'msg' => $msg ?: 'Proibido!'
		]);
	}

	public function sendNotFound($msg = false) {
		$this->setStatus(404);
		$this->sendResponse([
			'status' => 'NOT_FOUND',
			'msg' => $msg ?: 'Não encontrado'
		]);
	}

	public function sendBanned($msg = false) {
		$this->setStatus(402);
		$this->sendResponse([
			'status' => 'BANNED',
			'msg' => $msg ?: 'Usuário banido!'
		]);
	}

	public function gatherUserData() {
		if (!isset($_SESSION['user_data']) || empty($_SESSION['user_data'])) {
			return false;
		} else {
			$this->setUserData($_SESSION['user_data']);
			return true;
		}
	}

	public function forgetUserData() {
		unset($_SESSION['user_data']);
	}

	public function sendResponse($data) {
		$this->setHeadersOnce();
		http_response_code($this->responseStatus);
		echo json_encode($data, JSON_NUMERIC_CHECK);
		die;
	}

	public function isHeadersSetAlready() {
		return (bool) $this->headersSet;
	}

	public function setHeadersOnce() {
		if (!$this->isHeadersSetAlready()) {
			JsonCommunicator::setHeaders();
			$this->headersSet = true;
		}
	}

	public function setStatus($code) {
		$this->responseStatus = $code;
	}

	protected function setUserData($user_data) {
		$this->userData = $user_data;
	}

	protected function setRequest() {
		$data = $_REQUEST;
		try {
			$output = file_get_contents("php://input");
			json_decode($output);
			if (json_last_error() == JSON_ERROR_NONE && trim($output) != '') {
				$data = array_merge($data, json_decode($output, true));
			}
		} catch (Throwable $e) { var_dump($e); $this->sendBadRequest('Opa! Eu não conheço esse erro...'); }
		$this->request = $data;
	}

	public function getRequest() {
		return $this->request;
	}

	public function getUserData() {
		return $this->userData;
	}

	public function userId() {
		return $this->userData['id'] ?: null;
	}

	public function userName() {
		return $this->userData['username'] ?: null;
	}

	public function userFullName() {
		return $this->userData['name'] ?: null;
	}

	public function getArg($arg, $protect = true) {
		if (array_key_exists($arg, $this->request)) {
			$output = $this->request[$arg];
			try {
				json_decode(trim($output));
				if (json_last_error() == JSON_ERROR_NONE && trim($output) != '') {
					return json_decode($output);
				}
			} catch (Throwable $e) { /* Vamos ignorar aqui */ }
			
			if ($protect) {
				return pg_escape_string($output);
			} else {
				return $output;
			}
		} else {
			return null;
		}
	}

	static function respond($data, $code = 200) {
		JsonCommunicator::setHeaders();
		http_response_code($code);
		echo json_encode($data);
		die;
	}

	static function compare($a, $b) {
		//echo strcmp(intval(str_replace("'", '',$b["time"])), intval(str_replace("'", '',$a["time"])));
		$f = 0;
		if (intval(str_replace("'", '',$b["time"])) < intval(str_replace("'", '',$a["time"]))) {
			$f = -1;
		} elseif (intval(str_replace("'", '',$a["time"])) < intval(str_replace("'", '',$b["time"]))) {
			$f = 1;
		}
		
		return $f;
	}
	
	static function setHeaders() {
		$httpOrigin = (isset($_SERVER['HTTP_ORIGIN']) && !empty($_SERVER['HTTP_ORIGIN'])) ? $_SERVER['HTTP_ORIGIN'] : '';
		header('Content-Type: application/json');
		header("Access-Control-Allow-Origin:  https://sandbox.pagseguro.uol.com.br");//.$httpOrigin);
		header('Access-Control-Allow-Credentials: true');
		header('Access-Control-Allow-Methods: GET, PUT, POST, DELETE, OPTIONS');
		header('Access-Control-Max-Age: 1000');
		header('Access-Control-Allow-Headers: Content-Type, Content-Range, Content-Disposition, Content-Description, X-Requested-With');
	}
}


set_error_handler(function($errno, $ex) { 
	$return_data = [];
	$return_data['status'] = 'ERROR';
	$return_data['msg'] = $ex;
	$return_data['errno'] = $errno;
	$return_data['errdata'] = 'Crash';
	JsonCommunicator::respond($return_data, 500);
});

set_exception_handler(function($ex) { 
	$return_data = [];
	$return_data['status'] = 'EXCEPTION';
	$return_data['msg'] = $ex->getMessage();
	$return_data['errno'] = $ex->getCode();
	$return_data['errdata'] = 'Crash';
	JsonCommunicator::respond($return_data, 500);
});
?>